from django import forms
from .models import MaterialConsumption, MaterialConsumptionItem, Product, StockMovement, Warehouse, ProductCategory, MaterialType
from django.forms.models import inlineformset_factory

class ProductForm(forms.ModelForm):
    class Meta:
        model = Product
        fields = [
            'product_code', 'name', 'material_type', 'description',
            'unit_of_measurement', 'min_stock_level'
        ]
        exclude = ['created_by', 'created_at', 'current_stock']
        widgets = {
            'description': forms.Textarea(attrs={'rows': 3, 'class': 'form-control'}),
            'unit_of_measurement': forms.Select(attrs={'class': 'form-select'}),
            'material_type': forms.Select(attrs={'class': 'form-select'}),
            'min_stock_level': forms.NumberInput(attrs={'min': 0, 'class': 'form-control'}),
        }
        labels = {
            'product_code': 'Product Code',
            'material_type': 'Material Type',
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['material_type'].queryset = MaterialType.objects.all()
        self.fields['product_code'].widget.attrs.update({'placeholder': 'e.g., PROD-001'})

class StockMovementForm(forms.ModelForm):
    class Meta:
        model = StockMovement
        fields = '__all__'
        widgets = {
            'product': forms.Select(attrs={'class': 'form-select'}),
            'warehouse': forms.Select(attrs={'class': 'form-select'}),
            'movement_type': forms.Select(attrs={'class': 'form-select'}),
            'quantity': forms.NumberInput(attrs={'step': '0.001'}),
            'purchase_price': forms.NumberInput(attrs={'step': '0.01'}),
            'sale_price': forms.NumberInput(attrs={'step': '0.01'}),
            'notes': forms.Textarea(attrs={'rows': 3}),
        }
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['user'].initial = self.initial.get('user') or (
            self.instance.user if self.instance else None
        )


class MaterialConsumptionForm(forms.ModelForm):
    warehouse = forms.ModelChoiceField(queryset=Warehouse.objects.all(),label="Warehouse",required=True)

    class Meta:
        model = MaterialConsumption
        fields = ['project', 'date', 'remarks']
        widgets = {
            'project': forms.Select(attrs={'class': 'form-select'}),
            'date': forms.DateInput(attrs={'type': 'date', 'class': 'form-control'}),            
            'remarks': forms.Textarea(attrs={'class': 'form-control', 'rows': 2, 'style': 'height: auto;'}),

        }
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['warehouse'].widget.attrs.update({'class': 'form-control'})
        


class MaterialConsumptionItemForm(forms.ModelForm):
    class Meta:
        model = MaterialConsumptionItem
        fields = ['product', 'quantity', 'unit_price']
        widgets = {
            'product': forms.Select(attrs={'class': 'form-control'}),
            'unit_price': forms.NumberInput(attrs={'class': 'form-control', 'step': '0.01'}),
            'quantity': forms.NumberInput(attrs={'class': 'form-control'}),
        }


MaterialConsumptionItemFormSet = inlineformset_factory(
    MaterialConsumption,
    MaterialConsumptionItem,
    form=MaterialConsumptionItemForm,
    extra=1,
    can_delete=True
)
    