# In management/commands/setup_chart_of_accounts.py
from django.core.management.base import BaseCommand
from accounting.models import AccountCategory, AccountSubType, Account

class Command(BaseCommand):
    help = 'Sets up a standard chart of accounts'
    
    def handle(self, *args, **options):
        # Create account categories
        categories = [
            ('Asset', 'debit'),
            ('Liability', 'credit'),
            ('Equity', 'credit'),
            ('Revenue', 'credit'),
            ('Expense', 'debit'),
        ]
        
        for name, normal_balance in categories:
            AccountCategory.objects.get_or_create(
                name=name,
                defaults={'normal_balance': normal_balance}
            )
        
        # Create sub-types
        sub_types = [
            ('Asset', 'Current Assets'),
            ('Asset', 'Fixed Assets'),
            ('Liability', 'Current Liabilities'),
            ('Liability', 'Long-Term Liabilities'),
            ('Equity', 'Owner\'s Equity'),
            ('Revenue', 'Operating Revenue'),
            ('Revenue', 'Other Revenue'),
            ('Expense', 'Cost of Goods Sold'),
            ('Expense', 'Operating Expenses'),
        ]
        
        for category_name, sub_type_name in sub_types:
            category = AccountCategory.objects.get(name=category_name)
            AccountSubType.objects.get_or_create(
                category=category,
                name=sub_type_name
            )
        
        # Create base accounts
        accounts = [
            # Assets
            ('1010', 'Cash on Hand', 'Current Assets'),
            ('1020', 'Bank Account', 'Current Assets'),
            ('1100', 'Accounts Receivable', 'Current Assets'),
            ('1400', 'Inventory', 'Current Assets'),
            ('1500', 'Equipment', 'Fixed Assets'),
            
            # Liabilities
            ('2010', 'Accounts Payable', 'Current Liabilities'),
            ('2200', 'Sales Tax Payable', 'Current Liabilities'),
            
            # Equity
            ('3010', 'Owner\'s Capital', 'Owner\'s Equity'),
            ('3020', 'Retained Earnings', 'Owner\'s Equity'),
            
            # Revenue
            ('4010', 'Product Sales', 'Operating Revenue'),
            ('4020', 'Service Revenue', 'Operating Revenue'),
            
            # Expenses
            ('5010', 'Cost of Goods Sold', 'Cost of Goods Sold'),
            ('5020', 'Shipping Costs', 'Cost of Goods Sold'),
            ('6010', 'Rent Expense', 'Operating Expenses'),
            ('6020', 'Utilities', 'Operating Expenses'),
            ('6030', 'Salaries', 'Operating Expenses'),
        ]
        
        for code, name, sub_type_name in accounts:
            sub_type = AccountSubType.objects.get(name=sub_type_name)
            Account.objects.get_or_create(
                code=code,
                defaults={
                    'name': name,
                    'sub_type': sub_type
                }
            )
        
        self.stdout.write(self.style.SUCCESS('Successfully set up chart of accounts'))